#ifndef _WIN32
#include <unistd.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <sys/types.h>

#include "libfma.h"
#include "lf_internal.h"
#include "lf_fabric.h"
#include "lf_switch.h"
#include "lf_product_def.h"
#include "lf_wget.h"

#define SW16_MAX_SLOTS 16

#ifdef REESE
void
rfprint(FILE *fp, char **w, int wc)
{
  int i;
  if (fp == NULL) return;

  for (i=0; i<wc; ++i) {
    fprintf(fp, "%s ", w[i]);
  }
  fprintf(fp, "\n");
  fflush(fp);
}
#endif

/*
 * Local data structures
 */
struct sw16_encl_desc {
  char *product_id;
  int num_bp_slots;
};
static struct sw16_encl_desc sw16_encl_info[] = {
  { "M3-E16", 0 },
  { "M3-E32", 2 },
  { "M3-E64", 4 },
  { "M3-E128", 8 },
  { NULL, 0 }
};

/* helper macro for loop below */
#define PARSE_VALUE(D, S, V)				\
  else if (strcmp(wp[0], (S)) == 0) {			\
    if ((D) == NULL) {					\
      badstr = (S);					\
    } else {						\
      (D)->new_vals.V = atoll(wp[1]);			\
    }							\
  }

int
lf_query_switch_x16_html(
  lf_enclosure_t *ep)
{
  FILE *fp;
  char buf[LF_STRING_LEN];
  char *wp[LF_STRING_LEN];
  int rc;
  int wc;
  int userslot;
  int slot;
  int port;
  int port_is_xbar;
  int xbarportno;
  struct lf_linecard *lp;
  struct lf_linecard_data *ldp;
  struct lf_xbarport_data *xdp;
  struct lf_xcvr_data *xcdp;
  struct lf_xbar *xp;
  char *badstr;
  int temp_index;
  int retry;
#ifdef REESE
FILE *dfp;
{
  lf_string_t name, name2, cmd;
  system("mkdir -p /tmp/switch");
  sprintf(name, "/tmp/switch/%s", ep->name);
  sprintf(name2, "/tmp/switch/%s.old", ep->name);
  sprintf(cmd, "mv %s %s", name, name2);
  system(cmd);
  dfp  = fopen(name, "w");
}
#endif

  retry = FALSE;	/* by default, not a retryable error */
  port_is_xbar = 0;	/* init stuff to suppress warnings */
  ldp = NULL;
  xp = NULL;
  temp_index = 0;

  /* open and read all switch components */
  fp = lf_wget_open(ep->name, "/all");
  if (fp == NULL) {
    LF_ERROR(("Error opening stream to %s", ep->name));
  }

  /*
   * Initialize some things
   */
  xbarportno = 1;
  port = -1;
  lp = NULL;
  userslot = -1;
  slot = -1;
  xcdp = NULL;
  xdp = NULL;

  /* 
   * Read the whole file.  This approach to parsing the stream just remembers
   * the last slot, port number, and port type to know what to do with data
   */
  while (lf_wget_next_line(fp, buf, wp, &wc) != NULL) {
#ifdef REESE
rfprint(dfp, wp, wc);
#endif
    badstr = NULL;

    /* this first batch of words are the control words */

    /* remember last slot number */
    if (strcmp(wp[0], "Slot") == 0) {

      userslot = atoi(wp[1]);
      if (userslot >= ep->lc_slotbase
	  && userslot < ep->lc_slotbase + ep->num_lc_slots) {
	slot = userslot - ep->lc_slotbase;
	lp = ep->slots[slot];
	temp_index = 0;

	if (lp != NULL) {
	  ldp = lp->data;
	  ldp->seen = TRUE;	/* mark this slot as seen */
	}

      } else if (userslot >= ep->bp_slotbase
	         && userslot < ep->bp_slotbase + ep->num_bp_slots) {
	slot = userslot - ep->bp_slotbase + ep->num_lc_slots;
	lp = ep->slots[slot];
	if (lp != NULL) {
	  ldp = lp->data;
	  ldp->seen = TRUE;	/* mark this slot as seen */
	  temp_index = 0;
#if 0 /* permit backplane slots to be discovered */
	} else {
	  retry = TRUE;
	  LF_ERROR(("unexpected backplane slot in %s[%d]", ep->name, userslot));
#endif
	}

      } else {
	lp = NULL;
	temp_index = 0;
	slot = -1;
      }

    /* XbarPort is also used to sanity-check the switch outout */
    } else if (strcmp(wp[0], "XbarPort") == 0) {
      int xpn;
      xpn = atoi(wp[1]);
      if (xpn != xbarportno) {
	retry = TRUE;
#if 0
	LF_ERROR(("%s: wrong xbarport number: %d, expected %d",
		   ep->name, xpn, xbarportno));
#else 
	goto except;
#endif
      }
      ++xbarportno;

    /* got serial number and product id */
    } else if (wc > 2 && strcmp(wp[0], "serialNumber") == 0) {
      char *serial_no;
      char *product_id;

      serial_no = wp[1];
      product_id = lf_product_id_alias(wp[2]);

      /*
       * This is the time to check for a new or changed  linecard.
       * slot is used to keep from doing this twice since serial number can
       * be in the stream twice
       */
      if (slot >= 0) {

	/* if no lp, but slot is good, found a new linecard */
	if (lp == NULL) {
	  rc = lf_note_new_linecard(ep, slot, product_id, serial_no);
	  if (rc == -1) {
	    LF_ERROR(("error noticing new linecard in %s[%d]",
		       ep->name, userslot));
	  }

	/*
	 * lp is good- First, fill in serial no if it is missing, then if
	 * either serial_no or product id don't match, it's a different card
	 */
	} else {

	  /*if no serial number, set it */
	  if (lp->serial_no == NULL || lp->serial_no[0] == '\0') {
	    LF_FREE(lp->serial_no);
	    LF_DUP_STRING(lp->serial_no, serial_no);
	  }

	  /* if serial number or product ID different, card changed */
	  if (strcmp(lp->serial_no, serial_no) != 0
	      || strcmp(lp->product_id, product_id) != 0) {
	  
	    /* require a card to be "changed" several times in a row */
	    if (++ldp->change_cnt < 3) {
	      retry = TRUE;
#if 0
	      LF_ERROR(("Linecard changed, retrying"));
#else 
	      goto except;
#endif
	    }

	    rc = lf_note_new_linecard(ep, lp->slot, product_id, serial_no);
	    if (rc == -1) {
	      LF_ERROR(("error noticing changed linecard in %s[%d]",
			 ep->name, userslot));
	    }
	  }
	}
	slot = -1;	/* serial number shows up twice - who knows why?? */
      }

    } else if (lp == NULL) {	/* all else needs a linecard */
      continue;

    /* mark start of an xcvr stanza? */
    } else if (lp->num_xcvrs > 0 
               && strcmp(wp[0], lp->def->xcvr_port_string) == 0) {
      port_is_xbar = FALSE;

    /* we are now looking at xbar ports */
    } else if (strcmp(wp[0], "Xbar16") == 0) {
      port_is_xbar = TRUE;
      if (lp->num_xbars < 1) {
	retry = TRUE;
#if 0
	LF_ERROR(("unexpected xbar port on slot %d", userslot));
#else
	goto except;
#endif
      }
      xp = LF_XBAR(lp->xbars[0]);

    /* snag a port number */
    } else if (strcmp(wp[0], "portNumber") == 0) {
      port = atoi(wp[1]);
      if (port_is_xbar) {
	if (port < 0 || port >= xp->num_ports) {
	  retry = TRUE;
#if 0
	  LF_ERROR(("bad xbar port number %d", port));
#else
	  goto except;
#endif
	}
	xdp = xp->data + port;

      } else {
	if (port < 0 || port >= lp->num_xcvrs) {
	  retry = TRUE;
#if 0
	  LF_ERROR(("bad transceiver number %d", port));
#else 
	  goto except;
#endif
	}
	xcdp = LF_XCVR(lp->xcvrs[port])->data;
      }

    /* Get temperature index */
    } else if (lp != NULL && strcmp(wp[0], "temperature") == 0 && wc == 5) {
      if (temp_index < 0 || temp_index >= LF_SWITCH_NUM_TEMPS) {
	retry = TRUE;
#if 0
	LF_ERROR(("Error: temp index is %d for slot %d",
	      temp_index, lf_slot_display_no(lp)));
#else
	goto except;
#endif
      }

      /* Skip over '(' in the temp */
      if (wp[3][0] == '(') ++wp[3];
      ldp->new_vals.temperature[temp_index] = atoi(wp[3]);
      ++temp_index;

    /* Get overtemp count */
    } else if (lp != NULL && strcmp(wp[0], "overTemperatureCount") == 0) {
      ldp->new_vals.overtempcount = atoi(wp[1]);

    /* make sure there are at least 2 words to continue */
    } else if (wc < 2) {
      /* do nothing */
    }

    /* This section is where we get data values and save them away */

    /* xbar values */
    PARSE_VALUE(xdp, "goodCrcs", goodcrcs)
    PARSE_VALUE(xdp, "badCrcs", badcrcs)
    PARSE_VALUE(xdp, "invalidRoutes", invalidroutes)
    PARSE_VALUE(xdp, "receiveTimeoutCount", receivetimeoutcount)
    PARSE_VALUE(xdp, "transmitTimeoutCount", transmittimeoutcount)
    PARSE_VALUE(xdp, "controlForXbarPort", control)
    PARSE_VALUE(xdp, "portDown", portdown)
    PARSE_VALUE(xdp, "portUpCount", portflipcount)

    PARSE_VALUE(xcdp, "signalLost", signallost)
    PARSE_VALUE(xcdp, "signalLostCount", signallostcount)
    PARSE_VALUE(xcdp, "controlForShortwaveFiberPort", control)

    /* report delayed error */
    if (badstr != NULL) {
      retry = TRUE;
#if 0
      LF_ERROR(("Unexpected keyword %s", badstr));
#else
      goto except;
#endif
    }
  }

  lf_wget_close(fp);
  fp = NULL;

  /* make sure we got at least SOMETHING */
  if (userslot == -1) {
    LF_ERROR(("No data received from switch %s", ep->name));
  }

#ifdef REESE
if (dfp != NULL) fclose(dfp);
#endif
  return LF_QUERY_OK;

 except:
  if (fp != NULL) lf_wget_close(fp);
#ifdef REESE
if (dfp != NULL) { fclose(dfp); printf("written\n"); }
#endif
  if (retry) {
    return LF_QUERY_RETRY;
  } else {
    return LF_QUERY_FATAL;
  }
}

/*
 * Determine the switch type and fill in switch info if it is
 * an x16 based switch.
 */
int
lf_probe_switch_x16(
struct lf_enclosure *ep)
{
  FILE *fp;
  char buf[LF_STRING_LEN];
  char *wp[LF_STRING_LEN];
  char *rv;
  int rc;
  int wc;
  int slot;
  int num_bp_slots;
  char *slot_type[SW16_MAX_SLOTS];
  struct sw16_encl_desc *eip;

  /* init for cleanup */
  fp = NULL;

  /* open a stream to the switch */
  fp = lf_wget_open(ep->name, "/");
  if (fp == NULL) {
    return -1;
  }

  /* first line should be "Myrinet Switch" */
  rv = lf_wget_next_line(fp, buf, wp, &wc);
  if (rv == NULL
      || wc < 2
      || strcmp(wp[0], "Myrinet") != 0
      || strcmp(wp[1], "Switch") != 0) {
    goto except;
  }

  /*
   * look up and record each slot type.  count backplane slots
   */ 
  num_bp_slots = 0;
  memset(slot_type, 0, sizeof(slot_type));

  while (lf_wget_next_line(fp, buf, wp, &wc) != NULL) {
    int woff;

    /* sometimes the word "All" sneaks on the line, skip over it */
    if (strcmp(wp[0], "All") == 0) {
      woff = 1;
    } else {
      woff = 0;
    }

    if (wc > (2+woff) && strcmp(wp[0+woff], "Slot") == 0) {
      char *stype;

      slot = atoi(wp[1+woff]);
      if (slot <= 40) {		/* ignore the fan, etc. slots */
	char *ap;

	/* remove any parens around string */
	ap = wp[2+woff];
	if (*ap == '(') ++ap;
	if (ap[strlen(ap)-1] == ')') ap[strlen(ap)-1] = '\0';

	stype = lf_product_id_alias(ap);
	if (strcmp(stype, "BP-SW16") == 0) {
	  ++num_bp_slots;
	} else if (slot > 0 && slot <= SW16_MAX_SLOTS) {
	  slot_type[slot-1] = stype;		/* slots nos are 1-based */
	}
	if (lf_get_linecard_definition(stype) == NULL) {
	  LF_ERROR(("bad linecard type \"%s\"", wp[2+woff]));
	}
      }
    }
  }

  lf_wget_close(fp);		/* done reading switch */
  fp = NULL;

  /* 
   * Find the enclosure type based on number of backplane slots
   */
  eip = sw16_encl_info;
  while (eip->product_id != NULL) {
    if (eip->num_bp_slots == num_bp_slots) {
      break;
    }
    ++eip;
  }
  if (eip->product_id == NULL) {
    LF_ERROR(("bad number of backplane slots (%d)", num_bp_slots));
  }
  
  /*
   * Allocate and fill in the enclosure struct and the slots
   */
  LF_DUP_STRING(ep->product_id, eip->product_id);
  rc = lf_fill_enclosure(ep);
  if (rc == -1) goto except;

  return 0;

 except:
  if (fp != NULL) lf_wget_close(fp);
  return -1;
}
